<?php

namespace App\Models;

use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Support\Str;
use Illuminate\Database\Eloquent\Factories\HasFactory;
class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    protected $primaryKey = 'uid';
    public $incrementing = false;
    protected $keyType = 'string';

    protected $fillable = [
        'uid', 'email', 'name', 'address', 'display_photo_url', 'membership_tier',
        'assigned_trainer_id', 'assigned_doctor_id', 'wallet_balance', 'dob',
        'is_active', 'credit_balance', 'allergies', 'password',
    ];

    protected $hidden = [
        'password',
        'remember_token',
    ];

    protected $casts = [
        'allergies' => 'array',
        'wallet_balance' => 'decimal:2',
        'credit_balance' => 'decimal:2',
        'is_active' => 'boolean',
        'dob' => 'date',
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
    ];

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($model) {
            if (!$model->uid) {
                $model->uid = (string) Str::uuid();
            }
        });
    }

    public function roles()
    {
        return $this->belongsToMany(Role::class, 'user_roles', 'user_id', 'role_id');
    }

    public function hasRole($role)
    {
        return $this->roles()->where('slug', $role)->exists();
    }

    public function assignRole($role)
    {
        if (is_string($role)) {
            $role = Role::where('slug', $role)->firstOrFail();
        }
        $this->roles()->syncWithoutDetaching($role);
    }

    public function removeRole($role)
    {
        if (is_string($role)) {
            $role = Role::where('slug', $role)->firstOrFail();
        }
        $this->roles()->detach($role);
    }

    public function trainer()
    {
        return $this->belongsTo(Trainer::class, 'assigned_trainer_id', 'trainer_id');
    }

    public function doctor()
    {
        return $this->belongsTo(Doctor::class, 'assigned_doctor_id', 'doctor_id');
    }

    public function healthReports()
    {
        return $this->hasMany(HealthReport::class, 'user_id', 'uid');
    }

    public function groceryOrders()
    {
        return $this->hasMany(GroceryOrder::class, 'user_id', 'uid');
    }
    public function mealPlans()
    {
        return $this->hasMany(MealPlan::class, 'user_id', 'uid');
    }

    public function assignedMealPlans()
    {
        return $this->hasMany(MealPlan::class, 'assigned_by', 'uid');
    }


    public function assignedUsers()
    {
        return $this->belongsToMany(User::class, 'trainer_user', 'trainer_id', 'user_id', 'uid', 'uid');
    }


    public function chats()
    {
        return $this->hasMany(Chat::class, 'user_id', 'uid');
    }
}