<?php

namespace App\Http\Controllers;

use App\Models\Trainer;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class TrainerController extends Controller
{
    public function index()
    {
        $trainers = Trainer::with('user')->where('is_active', true)->get();
        return response()->json($trainers);
    }

    public function store(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,uid',
            'specialization' => 'required|string',
            'bio' => 'required|string',
            'certification' => 'required|string'
        ]);

        $user = User::find($request->user_id);
        $user->assignRole('trainer');

        $trainer = Trainer::create($request->all());
        return response()->json($trainer->load('user'), 201);
    }

    public function show(Trainer $trainer)
    {
        return response()->json($trainer->load(['user', 'assignedUsers']));
    }

    public function update(Request $request, Trainer $trainer)
    {
        $validated = $request->validate([
            'specialization' => 'sometimes|string',
            'bio' => 'sometimes|string',
            'certification' => 'sometimes|string',
            'is_active' => 'sometimes|boolean'
        ]);

        $trainer->update($validated);
        return response()->json($trainer->load('user'));
    }

    public function assignUser(Request $request, Trainer $trainer)
    {
        $request->validate([
            'user_id' => 'required|exists:users,uid'
        ]);

        $user = User::find($request->user_id);
        $user->update(['assigned_trainer_id' => $trainer->trainer_id]);

        return response()->json([
            'message' => 'User assigned successfully',
            'trainer' => $trainer->load(['user', 'assignedUsers'])
        ]);
    }

    public function unassignUser(Request $request, Trainer $trainer)
    {
        $request->validate([
            'user_id' => 'required|exists:users,uid'
        ]);

        $user = User::find($request->user_id);
        if ($user->assigned_trainer_id === $trainer->trainer_id) {
            $user->update(['assigned_trainer_id' => null]);
        }

        return response()->json([
            'message' => 'User unassigned successfully',
            'trainer' => $trainer->load(['user', 'assignedUsers'])
        ]);
    }

    public function myClients(Request $request)
    {
        $trainer = Trainer::where('user_id', $request->user()->uid)->firstOrFail();
        $clients = $trainer->assignedUsers()->with(['healthReports', 'mealPlans'])->get();

        return response()->json($clients);
    }
} 