<?php

namespace App\Http\Controllers;

use App\Models\MealPlan;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class MealPlanController extends Controller
{
    public function index(Request $request)
    {
        $query = MealPlan::query();

        // If user is not admin/trainer, only show their meal plans
        if (!Auth::user()->hasRole(['super-admin', 'admin', 'trainer'])) {
            $query->where('user_id', Auth::user()->uid);
        } elseif (Auth::user()->hasRole('trainer')) {
            // Trainers can see meal plans of their assigned users
            $trainer = Auth::user()->trainer;
            if ($trainer) {
                $userIds = $trainer->assignedUsers()->pluck('uid');
                $query->whereIn('user_id', $userIds);
            }
        }

        if ($request->has('week_of')) {
            $query->where('week_of', $request->week_of);
        }

        $mealPlans = $query->with('user')->orderBy('week_of', 'desc')->get();
        return response()->json($mealPlans);
    }

    public function store(Request $request)
    {
        if (!Auth::user()->hasRole(['super-admin', 'admin', 'trainer'])) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $request->validate([
            'user_id' => 'required|exists:users,uid',
            'week_of' => 'required|date',
            'plan_url' => 'required|url',
            'nutrition_focus' => 'required|string'
        ]);

        // If trainer, verify they can create plan for this user
        if (Auth::user()->hasRole('trainer')) {
            $trainer = Auth::user()->trainer;
            $userIds = $trainer->assignedUsers()->pluck('uid')->toArray();
            if (!in_array($request->user_id, $userIds)) {
                return response()->json(['message' => 'You can only create meal plans for your assigned users'], 403);
            }
        }

        $mealPlan = MealPlan::create([
            'user_id' => $request->user_id,
            'week_of' => $request->week_of,
            'plan_url' => $request->plan_url,
            'assigned_by' => Auth::user()->uid,
            'nutrition_focus' => $request->nutrition_focus
        ]);

        return response()->json($mealPlan->load('user'), 201);
    }

    public function show(MealPlan $mealPlan)
    {
        // Check if user can view this meal plan
        if (!$this->canAccessMealPlan($mealPlan) || !Auth::user()->hasRole(['super-admin', 'admin', 'trainer'])) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        return response()->json($mealPlan->load('user'));
    }

    public function update(Request $request, MealPlan $mealPlan)
    {
        if (!$this->canAccessMealPlan($mealPlan) || !Auth::user()->hasRole(['super-admin', 'admin', 'trainer'])) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $validated = $request->validate([
            'week_of' => 'sometimes|date',
            'plan_url' => 'sometimes|url',
            'nutrition_focus' => 'sometimes|string'
        ]);

        $mealPlan->update($validated);

        return response()->json($mealPlan->load('user'));
    }

    public function destroy(MealPlan $mealPlan)
    {
        if (!$this->canAccessMealPlan($mealPlan) || !Auth::user()->hasRole(['super-admin', 'admin', 'trainer'])) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $mealPlan->delete();

        return response()->json(['message' => 'Meal plan deleted successfully']);
    }

    private function canAccessMealPlan(MealPlan $mealPlan)
    {
        // Admin can access all meal plans
        if (Auth::user()->hasRole('super-admin')) {
            return true;
        }

        // Users can access their own meal plans
        if ($mealPlan->user_id === Auth::user()->uid) {
            return true;
        }

        // Trainers can access meal plans of their assigned users
        if (Auth::user()->hasRole('trainer')) {
            $trainer = Auth::user()->trainer;
            if ($trainer) {
                $userIds = $trainer->assignedUsers()->pluck('uid')->toArray();
                return in_array($mealPlan->user_id, $userIds);
            }
        }

        return false;
    }
} 