<?php

namespace App\Http\Controllers;

use App\Models\HealthReport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class HealthReportController extends Controller
{
    public function index(Request $request)
    {
        $reports = HealthReport::where('user_id', $request->user()->uid)
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json($reports);
    }

    public function store(Request $request)
    {
        $request->validate([
            'report_name' => 'required|string',
            'report_url' => 'required|url',
            'visibility' => 'required|string',
            'report_type' => 'required|string',
            'notes' => 'nullable|string',
            'status' => 'required|in:draft,final,reviewed',
            'expires_at' => 'nullable|date',
            'tags' => 'nullable|array'
        ]);

        $report = HealthReport::create([
            'user_id' => $request->user_id ?? $request->user()->uid,
            'report_name' => $request->report_name,
            'report_url' => $request->report_url,
            'uploaded_at' => now(),
            'uploaded_by' => $request->user()->uid,
            'uploader_role' => $request->user()->roles()->first()->slug ?? 'user',
            'visibility' => $request->visibility,
            'report_type' => $request->report_type,
            'notes' => $request->notes,
            'status' => $request->status,
            'expires_at' => $request->expires_at,
            'tags' => $request->tags,
            'viewed_by_user' => false
        ]);

        return response()->json($report, 201);
    }

    public function show(HealthReport $report)
    {
        if ($report->user_id !== Auth::user()->uid && !Auth::user()->hasRole(['admin', 'super-admin'])) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        if (!$report->viewed_by_user && $report->user_id === Auth::id()) {
            $report->update(['viewed_by_user' => true]);
        }

        return response()->json($report);
    }

    public function update(Request $request, HealthReport $report)
    {
        if ($report->user_id !== Auth::user()->uid && !Auth::user()->hasRole(['admin', 'super-admin'])) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $validated = $request->validate([
            'report_name' => 'sometimes|string',
            'report_url' => 'sometimes|url',
            'visibility' => 'sometimes|string',
            'report_type' => 'sometimes|string',
            'notes' => 'nullable|string',
            'status' => 'sometimes|in:draft,final,reviewed',
            'expires_at' => 'nullable|date',
            'tags' => 'nullable|array'
        ]);

        $report->update($validated);

        return response()->json($report);
    }

    public function destroy(HealthReport $report)
    {
        if ($report->user_id !== Auth::user()->uid && !Auth::user()->hasRole(['admin', 'super-admin'])) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $report->delete();

        return response()->json(['message' => 'Report deleted successfully']);
    }
} 