<?php

namespace App\Http\Controllers;

use App\Models\GroceryItem;
use App\Models\GroceryOrder;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class GroceryController extends Controller
{
    // Grocery Items Management
    public function listItems(Request $request)
    {
        $query = GroceryItem::query();
        
        if ($request->has('category')) {
            $query->where('category', $request->category);
        }

        if ($request->has('search')) {
            $query->where('name', 'like', '%' . $request->search . '%');
        }

        $items = $query->get();
        return response()->json($items);
    }

    public function createItem(Request $request)
    {
        if (!Auth::user()->hasRole(['super-admin', 'admin', 'personal-assistant'])) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $request->validate([
            'name' => 'required|string',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'category' => 'nullable|string',
            'nutritional_info' => 'nullable|array'
        ]);

        $item = GroceryItem::create($request->all());
        return response()->json($item, 201);
    }

    public function updateItem(Request $request, GroceryItem $item)
    {
        if (!Auth::user()->hasRole(['super-admin', 'admin', 'personal-assistant'])) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $validated = $request->validate([
            'name' => 'sometimes|string',
            'description' => 'nullable|string',
            'price' => 'sometimes|numeric|min:0',
            'category' => 'nullable|string',
            'nutritional_info' => 'nullable|array'
        ]);

        $item->update($validated);
        return response()->json($item);
    }

    // Grocery Orders Management
    public function listOrders(Request $request)
    {
        $query = GroceryOrder::with(['items', 'user']);

        // If user is not admin/PA, only show their orders
        if (!Auth::user()->hasRole(['super-admin', 'admin', 'personal-assistant'])) {
            $query->where('user_id', Auth::id());
        }

        if ($request->has('status')) {
            $query->where('status', $request->status);
        }

        $orders = $query->orderBy('created_at', 'desc')->get();
        return response()->json($orders);
    }

    public function createOrder(Request $request)
    {
        $request->validate([
            'items' => 'required|array',
            'items.*.item_id' => 'required|exists:grocery_items,id',
            'items.*.quantity' => 'required|integer|min:1',
            'pdf_url' => 'nullable|url',
            'tags' => 'nullable|array'
        ]);

        try {
            DB::beginTransaction();

            $total = 0;
            foreach ($request->items as $item) {
                $groceryItem = GroceryItem::find($item['item_id']);
                $total += $groceryItem->price * $item['quantity'];
            }

            $order = GroceryOrder::create([
                'user_id' => $request->user()->uid,
                'invoice_amount' => $total,
                'status' => 'pending',
                'pdf_url' => $request->pdf_url,
                'uploaded_by' => $request->user()->uid,
                'uploader_role' => $request->user()->roles()->first()->slug ?? 'user',
                'uploaded_at' => now(),
                'tags' => $request->tags
            ]);

            foreach ($request->items as $item) {
                $groceryItem = GroceryItem::find($item['item_id']);
                $order->items()->attach($item['item_id'], [
                    'quantity' => $item['quantity'],
                    'unit_price' => $groceryItem->price
                ]);
            }

            DB::commit();
            return response()->json($order->load('items'), 201);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Error creating order: ' . $e->getMessage()], 500);
        }
    }

    public function showOrder(GroceryOrder $order)
    {
        if ($order->user_id !== Auth::user()->uid && !Auth::user()->hasRole(['admin', 'personal-assistant'])) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        return response()->json($order->load(['items', 'user']));
    }

    public function updateOrderStatus(Request $request, GroceryOrder $order)
    {
        if ($order->user_id !== Auth::user()->uid && !Auth::user()->hasRole(['admin', 'personal-assistant'])) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $request->validate([
            'status' => 'required|in:pending,needs_revision,approved,purchased,cancelled',
            'user_feedback' => 'nullable|string',
            'purchase_note' => 'nullable|string'
        ]);

        try {
            DB::beginTransaction();

            $updateData = ['status' => $request->status];

            if ($request->status === 'needs_revision' && $request->user_feedback) {
                $updateData['user_feedback'] = $request->user_feedback;
                $updateData['user_feedback_at'] = now();
            }

            if ($request->status === 'approved') {
                $updateData['user_approved'] = true;
                $updateData['user_approved_at'] = now();
            }

            if ($request->status === 'purchased') {
                $updateData['purchased_at'] = now();
                $updateData['purchase_note'] = $request->purchase_note;
                
                // Handle wallet deduction
                if (!$order->wallet_deducted) {
                    $user = $order->user;
                    if ($user->wallet_balance >= $order->invoice_amount) {
                        $user->wallet_balance -= $order->invoice_amount;
                        $user->save();
                        $updateData['wallet_deducted'] = true;
                    } else {
                        DB::rollBack();
                        return response()->json(['message' => 'Insufficient wallet balance'], 400);
                    }
                }
            }

            $order->update($updateData);
            DB::commit();

            return response()->json($order->load(['items', 'user']));

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Error updating order: ' . $e->getMessage()], 500);
        }
    }

    public function addItemsToOrder(Request $request, GroceryOrder $order)
    {
        if ($order->status !== 'pending' && $order->status !== 'needs_revision') {
            return response()->json(['message' => 'Cannot modify order in current status'], 400);
        }

        $request->validate([
            'items' => 'required|array',
            'items.*.item_id' => 'required|exists:grocery_items,id',
            'items.*.quantity' => 'required|integer|min:1'
        ]);

        try {
            DB::beginTransaction();

            $additionalTotal = 0;
            foreach ($request->items as $item) {
                $groceryItem = GroceryItem::find($item['item_id']);
                $additionalTotal += $groceryItem->price * $item['quantity'];
                
                $order->items()->attach($item['item_id'], [
                    'quantity' => $item['quantity'],
                    'unit_price' => $groceryItem->price
                ]);
            }

            $order->invoice_amount += $additionalTotal;
            $order->save();

            DB::commit();
            return response()->json($order->load('items'));

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Error adding items to order'], 500);
        }

    }

    public function removeItemFromOrder(Request $request, GroceryOrder $order, GroceryItem $item)
    {
        if ($order->status !== 'pending' && $order->status !== 'needs_revision') {
            return response()->json(['message' => 'Cannot modify order in current status'], 400);
        }

        try {
            DB::beginTransaction();

            $orderItem = $order->items()->where('grocery_items.id', $item->id)->first();
            if ($orderItem) {
                $refundAmount = $orderItem->pivot->quantity * $orderItem->pivot->unit_price;
                $order->invoice_amount -= $refundAmount;
                $order->save();
                
                $order->items()->detach($item->id);
            }

            DB::commit();
            return response()->json($order->load('items'));

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['message' => 'Error removing item from order'], 500);
        }
    }

    
    
} 