<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class AuthController extends Controller
{
    public function login(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        $user = User::where('email', $request->email)->first();

        if (!$user || !Hash::check($request->password, $user->password)) {
            return response()->json(['message' => 'Invalid credentials'], 401);
        }

        $token = $user->createToken('auth_token')->plainTextToken;

        return response()->json([
            'user' => $user,
            'roles' => $user->roles()->pluck('name'),
            'token' => $token,
        ]);
    }

    public function register(Request $request)
    {
        $request->validate([
            'email' => 'required|email|unique:users',
            'name' => 'required|string',
            'password' => 'required|min:6',
            'membership_tier' => 'in:free,Origin,Accelerator,Sovereign',
            'role' => 'required|string|in:free-user,origin-user,accelerator-user,sovereign-user,trainer,doctor,personal-assistant,admin,super-admin',
        ]);

        $user = User::create([
            'email' => $request->email,
            'name' => $request->name,
            'password' => Hash::make($request->password),
            'membership_tier' => $request->membership_tier ?? 'free',
        ]);

        // Assign role based on membership tier if not explicitly provided
        $roleSlug = $request->role ?? match ($request->membership_tier) {
            'Origin' => 'origin-user',
            'Accelerator' => 'accelerator-user',
            'Sovereign' => 'sovereign-user',
            default => 'free-user',
        };

        $role = Role::where('slug', $roleSlug)->first();
        if ($role) {
            $user->assignRole($role);
        }

        $token = $user->createToken('auth_token')->plainTextToken;

        return response()->json([
            'user' => $user,
            'roles' => $user->roles()->pluck('name'),
            'token' => $token,
        ], 201);
    }

    // public function assignRole(Request $request)
    // {
    //     $request->validate([
    //         'user_id' => 'required|exists:users,uid',
    //         'role' => 'required|string|exists:roles,slug',
    //     ]);

    //     $user = User::findOrFail($request->user_id);
    //     $user->assignRole($request->role);

    //     return response()->json([
    //         'message' => 'Role assigned successfully',
    //         'user' => $user,
    //         'roles' => $user->roles()->pluck('name'),
    //     ]);
    // }
    public function assignRole(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,uid',
            'role' => 'required|string|exists:roles,slug',
        ]);

        // Check if the authenticated user is an admin or super-admin
        $user = auth()->user(); // Get the authenticated user
        if (!$user->hasRole('admin') && !$user->hasRole('super-admin')) {
            return response()->json(['message' => 'Unauthorized'], 403); // Unauthorized if not admin or super-admin
        }

        $userToAssign = User::findOrFail($request->user_id);
        $userToAssign->assignRole($request->role);

        return response()->json([
            'message' => 'Role assigned successfully',
            'user' => $userToAssign,
            'roles' => $userToAssign->roles()->pluck('name'),
        ]);
    }


    // public function removeRole(Request $request)
    // {
    //     $request->validate([
    //         'user_id' => 'required|exists:users,uid',
    //         'role' => 'required|string|exists:roles,slug',
    //     ]);

    //     $user = User::findOrFail($request->user_id);
    //     $user->removeRole($request->role);

    //     return response()->json([
    //         'message' => 'Role removed successfully',
    //         'user' => $user,
    //         'roles' => $user->roles()->pluck('name'),
    //     ]);
    // }

    public function removeRole(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,uid',
            'role' => 'required|string|exists:roles,slug',
        ]);

        // Check if the authenticated user is an admin or super-admin
        $user = auth()->user(); // Get the authenticated user
        if (!$user->hasRole('admin') && !$user->hasRole('super-admin')) {
            return response()->json(['message' => 'Unauthorized'], 403); // Unauthorized if not admin or super-admin
        }

        $userToRemoveRole = User::findOrFail($request->user_id);
        $userToRemoveRole->removeRole($request->role);

        return response()->json([
            'message' => 'Role removed successfully',
            'user' => $userToRemoveRole,
            'roles' => $userToRemoveRole->roles()->pluck('name'),
        ]);
    }

}